class MultipleSelectBoxes {
    constructor(prms) {
        try {
            this.selElmId = 0;
            this._mergeDefaultParams(prms);
            this._validateParams(prms);
            this._assignParamsToProps(prms);
            this._registerButtonEvents();
            this._appendInitialSelects();
        } catch (error) {
            console.error(error);
            alert(`Something error occured on initializing: ${error}`);
        }
    }
    _validateParams(prms) {
        let reqKeys = ['location', 'itemTexts', 'itemValues'];
        let jqKeys = ['location', 'removeButton', 'bindForm'];
        let arrKeys = ['itemTexts', 'itemValues', 'initialSelectedItems'];
        let numKeys = ['minShowingSelects', 'maxShowingSelects'];
        // 必須パラメータ
        for (let key of reqKeys)
            if (prms[key] === null)
                throw new Error(`The paramater '${key}' is required.`);
        // jQueryオブジェクトを受け取るパラメータ
        for (let key of jqKeys)
            if (prms[key] !== null && !(prms[key] instanceof jQuery))
                throw new Error(`The paramater '${key}' must be a jQuery object.`);
        // 配列を受け取るパラメータ
        for (let key of arrKeys)
            if (!Array.isArray(prms[key]))
                throw new Error(`The paramater '${key}' must be an array.`);
        // 数値を受け取るパラメータ
        for (let key of numKeys)
            if (isNaN(prms[key]) || prms[key] === true || prms[key] === false)
                throw new Error(`The paramater '${key}' must be an integer type.`);
        // 初期追加するselect要素数が設定した最大表示数を超えている
        let initCnt = prms.initialSelectedItems.length;
        let maxCnt = prms.maxShowingSelects;
        if (maxCnt != -1 && initCnt > maxCnt)
            throw new Error(`The length of paramater 'initialSelectedItems' exceeds 'maxShowingSelects'.`);
        for (let idx of prms.initialSelectedItems) {
            // 初期追加に指定された初期選択インデックスがアイテム配列に存在しない
            if (!prms.itemTexts.hasOwnProperty(idx))
                throw new Error(`The paramater 'initialSelectedItems' has an invalid index '${idx}' that not exists in 'itemTexts' or 'itemValues'.`);
            // 初期追加項目の中に未選択項目（空値）が含まれる場合、無視されるので警告だけ表示する
            if (idx == 0)
                console.warn(`The paramater 'initialSelectedItems' has an empty value index '${idx}' that is ignored on initial selects.`);
        }
        // アイテム配列の要素数がテキストと値で不一致
        if (prms.itemTexts.length !== prms.itemValues.length)
            throw new Error(`The length of paramaters between 'itemTexts' and 'itemValues' are different.`);
    }
    _mergeDefaultParams(prms) {
        let defPrms = {
            location: null,
            removeButton: null,
            bindForm: null,
            bindDelimitor: ',',
            itemTexts: null,
            itemValues: null,
            initialSelectedItems: [],
            minShowingSelects: 1,
            maxShowingSelects: -1,
            allowDuplicate: true,
            selectClassName: 'v-select-boxes',
            msgSelectsMaxExceeded: 'Max nodes that you can select has exceeded.',
            msgDuplicatedSelect: 'Items cannot be duplicated.',
            errorAlert: true
        };
        for (let key of Object.keys(defPrms))
            if (prms[key] === void 0)
                prms[key] = defPrms[key];
    }
    _assignParamsToProps(prms) {
        for (let key of Object.keys(prms))
            this[key] = prms[key];
    }
    _registerButtonEvents() {
        let removeButton = this.removeButton;
        if (removeButton !== void 0 && removeButton !== null)
            removeButton.on('click', this.removeSelect.bind(this));
    }
    _appendInitialSelects() {
        let initItems = this.initialSelectedItems;
        for (let i = 0; i < initItems.length; ++i)
            if (initItems[i] !== 0)     // 空値は無視
                this.appendSelect(initItems[i]);
        this._supplyEmptySelects();
        this._bindToForm();
    }
    getSelects() {
        let cname = this.selectClassName;
        return this.location.find(`.${cname}`);
    }
    getSelectsCount() {
        return this.getSelects().length;
    }
    getEmptySelectsCount() {
        let cnt = 0, _this = this;
        this.getSelects().each((i, sel) => {
            let value = $(sel).val();
            if (_this._isEmptyValue(value))
                ++cnt;
        });
        return cnt;
    }
    getLastSelect() {
        return this.getSelects().last();
    }
    getLastNotEmptySelect() {
        let sels = this.getSelects();
        for (let i = sels.length - 1; i > -1; --i) {
            let value = $(sels[i]).val()
            if (!this._isEmptyValue(value))
                return $(sels[i]);
        }
        return null;
    }
    getSelectedValues(rmEmp = true) {
        let values = [], _this = this;
        this.getSelects().each((i, sel) => {
            let value = $(sel).val();
            let isEmp = _this._isEmptyValue(value);
            if (isEmp && rmEmp)
                return;
            values.push(value);
        });
        return values;
    }
    _bindToForm() {
        if (this.bindForm === null)
            return;
        let values = this.getSelectedValues(true);
        let joined = values.join(this.bindDelimitor);
        this.bindForm.val(joined);
    }
    appendSelect(selectedIdx = 0) {
        let max = this.maxShowingSelects;
        let cnt = this.getSelectsCount();
        if (max != -1 && cnt >= max) {
            alert(this.msgSelectsMaxExceeded);
            return;
        }
        let html = this._makeSelectHtml(selectedIdx);
        this.location.append(html);
        let sel = this.getLastSelect().get(0);
        $(sel).change(this._onChange.bind(this));
        sel.selElmId = this.selElmId++;
        sel.beforeValue = this.itemValues[selectedIdx];
        this._bindToForm();
    }
    removeSelect() {
        let sel = this.getLastNotEmptySelect();
        if (sel !== null) {
            sel.remove();
            this._supplyEmptySelects()
        }
        this._bindToForm();
    }
    _supplyEmptySelects() {
        // 最低表示数を満たすまでselectを補充
        let cnt = this.getSelectsCount();
        let rest = this.minShowingSelects - cnt;
        let max = this.maxShowingSelects;
        for (let i = 0; i < rest; ++i)
            this.appendSelect();
        // 未選択項目が最低一つは存在するようにする
        let empCnt = this.getEmptySelectsCount();
        let isMax = max !== -1 && cnt == max;
        if (this.getEmptySelectsCount() === 0 && !isMax)
            this.appendSelect();
    }
    _onChange(e) {
        let target = e.target;
        let targetValue = $(target).val();
        if (this._isEmptyValue(targetValue)) {
            $(target).remove();
        } else if (!this.allowDuplicate && this._isDuplicated(target)) {
            alert(this.msgDuplicatedSelect);
            $(target).val(target.beforeValue);
            return;
        }
        target.beforeValue = targetValue;
        this._supplyEmptySelects();
        this._bindToForm();
    }
    _isDuplicated(target) {
        // 先頭項目(空値)の重複は無視
        if (target.selectedIndex === 0)
            return false;
        let targetValue = $(target).val();
        let duplicated = false;
        this.getSelects().each((i, sel) => {
            if (sel.selElmId === target.selElmId)
                return;
            if ($(sel).val() === targetValue) {
                duplicated = true;
                return false;
            }
        });
        return duplicated;
    }
    _isEmptyValue(value) {
        return value === this.itemValues[0];
    }
    _makeSelectHtml(selectedIdx = 0) {
        let html = `<select class="${this.selectClassName}">`;
        for (let i = 0; i < this.itemTexts.length; ++i) {
            let text = this.itemTexts[i];
            let value = this.itemValues[i];
            let attrSel = i === selectedIdx ? ' selected' : '';
            html += `<option value="${value}"${attrSel}>${text}</option>`;
        }
        html += `</select>`;
        return html;
    }
}